/*
 *                            COPYRIGHT
 *
 *  sch-rnd - modular/flexible schematics editor - Maxascii netlist export
 *  Copyright (C) 2025 Tibor 'Igor2' Palinkas
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA.
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */

#include <librnd/hid/hid.h>
#include <librnd/hid/hid_attrib.h>
#include <librnd/hid/hid_nogui.h>
#include <librnd/hid/hid_init.h>
#include <libcschem/project.h>
#include <libcschem/util_export.h>
#include <sch-rnd/export.h>

static const char maxascii_cookie[] = "Maxascii export hid";

static const rnd_export_opt_t maxascii_options[] = {
	{"outfile", "Name of the Maxascii netlist output file",
	 RND_HATT_STRING, 0, 0, {0, 0, 0}, 0},
#define HA_outfile 0

	{"view", "Name of the view to export (use first view when not specified)",
	 RND_HATT_STRING, 0, 0, {0, 0, 0}, 0},
#define HA_view 1
};

#define NUM_OPTIONS (sizeof(maxascii_options)/sizeof(maxascii_options[0]))

static rnd_hid_attr_val_t maxascii_values[NUM_OPTIONS];

static const rnd_export_opt_t *maxascii_get_export_options(rnd_hid_t *hid, int *n, rnd_design_t *dsg, void *appspec)
{
	const char *val = maxascii_values[HA_outfile].str;

	if ((dsg != NULL) && ((val == NULL) || (*val == '\0')))
		csch_derive_default_filename(dsg, 1, &maxascii_values[HA_outfile], ".net");

	if (n)
		*n = NUM_OPTIONS;
	return maxascii_options;
}

static void maxascii_do_export(rnd_hid_t *hid, rnd_design_t *design, rnd_hid_attr_val_t *options, void *appspec)
{
	csch_sheet_t *sheet = (csch_sheet_t *)design;
	int viewid = CSCH_VIEW_DEFAULT;

	if (!options) {
		maxascii_get_export_options(hid, 0, design, appspec);
		options = maxascii_values;
	}

	if ((options[HA_view].str != NULL) && (options[HA_view].str[0] != '\0')) {
		viewid = csch_view_get_id((csch_project_t *)sheet->hidlib.project, options[HA_view].str);
		if (viewid < 0) {
			rnd_message(RND_MSG_ERROR, "No such view in the project: '%s'\n", options[HA_view].str);
			return;
		}
	}

	sch_rnd_export_prj_abst((csch_project_t *)sheet->hidlib.project, sheet, viewid, "maxascii", options[HA_outfile].str, options);
}

static int maxascii_usage(rnd_hid_t *hid, const char *topic)
{
	fprintf(stderr, "\nMaxascii exporter command line arguments:\n\n");
	rnd_hid_usage(maxascii_options, sizeof(maxascii_options) / sizeof(maxascii_options[0]));
	fprintf(stderr, "\nUsage: sch-rnd [generic_options] -x maxascii [options] foo.rs\n\n");
	return 0;
}


static int maxascii_parse_arguments(rnd_hid_t *hid, int *argc, char ***argv)
{
	rnd_export_register_opts2(hid, maxascii_options, sizeof(maxascii_options) / sizeof(maxascii_options[0]), maxascii_cookie, 0);
	return rnd_hid_parse_command_line(argc, argv);
}

rnd_hid_t maxascii_hid = {0};
